//
//  XMapView.m
//  Version   : 1.2.5
//  Created by: Xu Liu
//  Email     : me@xuliu.info
//
//  This is one of class for clustering the pins in the MKMapView compoenent for everyone to use
//  under GPL license. When you use this class, please respect the author and keep author's
//  infomation in this class. If you need more information about this class, please visit
//  the website:http://www.xuliu.info/xMapView.html
//

#import "XMapView.h"
#define MERCATOR_RADIUS 85445659.44705395


@implementation XMapView
@synthesize xAnnotationContainer;
@synthesize xAnnotationsInMap;
@synthesize xAnnotationsInMapLocker;
@synthesize thresholdForRefreshTime;
@synthesize refreshIntervalForPins;
@synthesize needToHideGroups;
@synthesize clusteringRadius;
@synthesize enableClusteringRadiusAutoAdjust;
@synthesize clusteringGroupNrofPins;
@synthesize xdelegate;
@synthesize ignoreGroupTag;

-(void)initialize{
    //basic initialization
    xAnnotationContainer = [[NSMutableArray alloc] init];
    xAnnotationsInMap = [[NSMutableArray alloc] init];
    
    //total pins over than this value, the refresh speed of the map will increase from 0 to 1
    thresholdForRefreshTime = 200;
    xAnnotationsInMapLocker = [[NSLock alloc] init];
    
    //clustering radius
    clusteringRadius = 60;
    
    //auto adjust the radius based on a given pin
    enableClusteringRadiusAutoAdjust = YES;
    
    //set up the size of the clustering roup
    clusteringGroupNrofPins = 5;
    
    //the group name in this array, can not be shown
    needToHideGroups = [[NSMutableArray alloc] init];
    
    //igore group name to group all pins
    ignoreGroupTag = NO;
    

    
    //map view delegate
    self.delegate = self;
}


-(BOOL)PinsNeedToShow:(NSString *)clusteringGroupName
{
    BOOL __block showIt = YES;
    for(int i = 0; i < [needToHideGroups count]; i++)
    {
        NSString * strGroupName = (NSString *)[needToHideGroups objectAtIndex:i];
        if([clusteringGroupName compare:strGroupName] == 0){
            showIt = NO;
            break;
        }
    }
    return showIt;
}

-(void)reloadAnnotationsInMap{
    [xAnnotationsInMapLocker lock];
    
    //refresh xAnnotation
    [xAnnotationContainer enumerateObjectsUsingBlock:^(id obj, NSUInteger idx, BOOL *stop) {
        XAnnotation * annotation = (XAnnotation *)obj;
        [annotation updateXAnnotation];
    }];
    
    //remove XMapAnnotation
    [self performSelectorOnMainThread:@selector(removeAnnotations:) withObject:[self annotations] waitUntilDone:YES];
    
    [xAnnotationsInMapLocker unlock];
    
    //refresh map
    [self refreshAnnotationsInMap];
}

//refresh annotations in the map
-(void)refreshAnnotationsInMap{
    if(xAnnotationContainer != NULL)
    {
        dispatch_async(dispatch_get_main_queue(), ^{
            @synchronized(xAnnotationsInMap)
            {
                @synchronized(xAnnotationContainer)
                {
                    float durationOnDemanding = [xAnnotationContainer count] > thresholdForRefreshTime ? 1 : 0.5;
                    if(fabs([NSDate timeIntervalSinceReferenceDate] - refreshIntervalForPins) < durationOnDemanding){return;}
                    [xAnnotationsInMapLocker lock];
                    CLLocationCoordinate2D topLeft =   MKCoordinateForMapPoint(MKMapPointMake(self.visibleMapRect.origin.x, self.visibleMapRect.origin.y));
                    CLLocationCoordinate2D bottomRight =   MKCoordinateForMapPoint(MKMapPointMake(self.visibleMapRect.origin.x + self.visibleMapRect.size.width, self.visibleMapRect.origin.y + self.visibleMapRect.size.height));
                    NSMutableArray __block * arrayRemovable = [[NSMutableArray alloc] init];
                    NSMutableArray __block * arrayClustering = [[NSMutableArray alloc] init];
                    [xAnnotationContainer enumerateObjectsUsingBlock:^(id obj, NSUInteger idx, BOOL *stop) {
                        XAnnotation * xannoObj = (XAnnotation *)obj;
                        CLLocationCoordinate2D currentIconCoordinate = CLLocationCoordinate2DMake(xannoObj.latitude, xannoObj.longitude);
                        if(currentIconCoordinate.latitude >= bottomRight.latitude && currentIconCoordinate.latitude <= topLeft.latitude && currentIconCoordinate.longitude >= topLeft.longitude && currentIconCoordinate.longitude <= bottomRight.longitude)
                        {
                            if([self PinsNeedToShow:xannoObj.clustringGroupName]){
                                [arrayClustering addObject:xannoObj];
                            }
                            else{
                                [arrayRemovable addObject:xannoObj.uniqueId];
                            }
                        }
                        else{
                            [arrayRemovable addObject:xannoObj.uniqueId];
                        }
                    }];
                    
                    //based on map clear not useful annotation
                    NSArray * ans = self.annotations;
                    [ans enumerateObjectsUsingBlock:^(id objx, NSUInteger idxx, BOOL *stopx) {
                        if(objx != self.userLocation){
                            XMapAnnotation * ann = (XMapAnnotation *)objx;
                            BOOL __block isIn = NO;
                            [arrayClustering enumerateObjectsUsingBlock:^(id obj, NSUInteger idx, BOOL *stop) {
                                XAnnotation * xAnno = (XAnnotation *)obj;
                                if([ann.uniqueId compare:xAnno.uniqueId] == 0){
                                    isIn = YES;
                                    *stop = YES;
                                }
                            }];
                            if(!isIn){
                                if(ann.pinCategory != -1)// if it is not the clustering pin
                                {
                                    if(ann.uniqueId == nil){return;}
                                    [self XMapAnnotationRemover:@[ann.uniqueId]];
                                }
                                else{
                                    //clustering pin in the map
                                    //check the clustering pin is in the region
                                    if(ann.latitude >= bottomRight.latitude && ann.latitude <= topLeft.latitude && ann.longitude >= topLeft.longitude && ann.longitude <= bottomRight.longitude){
                                        //keep the clustering pin due to it is in the region
                                        //check whether this one is associcate with a pin group which need to be hide
                                        if(![self PinsNeedToShow:ann.clustringGroupName]){
                                            [self XMapAnnotationRemover:@[ann.uniqueId]];
                                        }
                                    }else{
                                        //remove clustering pin, because clustering pin is out of the map
                                        //here won't remove later updated new one clustering
                                        //need to clear the unupdated one in the clustering
                                        //process
                                        [self XMapAnnotationRemover:@[ann.uniqueId]];
                                    }
                                }
                            }
                        }
                    }];
                    
                    //remove non-useful icons, clear outside screen range
                    //avoid remove some other icon in the map
                    [self XMapAnnotationRemover:arrayRemovable];
                    [arrayRemovable removeAllObjects];
                    arrayRemovable = nil;
                    
                    //add useful icons
                    [self XMapAnnotationClustering:arrayClustering];
                    [arrayClustering removeAllObjects];
                    arrayClustering = nil;
                    refreshIntervalForPins = [NSDate timeIntervalSinceReferenceDate];
                }
            }
        });
    }
}

-(void)addXAnnotation:(XAnnotation *)annotation{
    [xAnnotationsInMapLocker lock];
    [xAnnotationContainer addObject:annotation];
    [xAnnotationsInMapLocker unlock];
    [self refreshAnnotationsInMap];
}


-(void)addXAnnotations:(NSArray *)annotations{
    [xAnnotationsInMapLocker lock];
    [xAnnotationContainer addObjectsFromArray:annotations];
    [xAnnotationsInMapLocker unlock];
    [self refreshAnnotationsInMap];
    
}

-(void)removeXAnnotationById:(NSString *)xannotationId{
    [xAnnotationsInMapLocker lock];
    for(int i = 0; i < [xAnnotationContainer count]; i++){
        XAnnotation * annotation = (XAnnotation *)[xAnnotationContainer objectAtIndex:i];
        if([annotation.uniqueId compare:xannotationId] == 0){
            [xAnnotationContainer removeObjectAtIndex:i];
        }
    }
    NSArray * annotations = [self annotations];
    for(int i = 0; i < [annotations count]; i++){
        if([[annotations objectAtIndex:i] isKindOfClass:[MKUserLocation class]]){continue;}
        XMapAnnotation * annotation = (XMapAnnotation *)[annotations objectAtIndex:i];
        if([annotation.uniqueId compare:xannotationId] == 0){
            [self removeAnnotation:annotation];
        }
    }
    [xAnnotationsInMapLocker unlock];
    [self refreshAnnotationsInMap];
}

-(void)removeAllXAnnotations{
    [xAnnotationsInMapLocker lock];
    [self.xAnnotationContainer removeAllObjects];
    [self removeAnnotations:[self annotations]];
    [xAnnotationsInMapLocker unlock];
    [self refreshAnnotationsInMap];
}

-(void)removeXAnnotations:(NSArray *)annotationIds{
    [xAnnotationsInMapLocker lock];
    for(int x = 0; x < [annotationIds count]; x++){
        NSString * xannotationId = [NSString stringWithFormat:@"%@",[annotationIds objectAtIndex:x]];
        for(int i = 0; i < [xAnnotationContainer count]; i++){
            XAnnotation * annotation = (XAnnotation *)[xAnnotationContainer objectAtIndex:i];
            if([annotation.uniqueId compare:xannotationId] == 0){
                [xAnnotationContainer removeObjectAtIndex:i];
            }
        }
    }
    
    NSArray * annotations = [self annotations];
    for(int x = 0; x < [annotationIds count]; x++){
        NSString * xannotationId = [NSString stringWithFormat:@"%@",[annotationIds objectAtIndex:x]];
        for(int i = 0; i < [annotations count]; i++){
            if([[annotations objectAtIndex:i] isKindOfClass:[MKUserLocation class]]){continue;}
            XMapAnnotation * annotation = (XMapAnnotation *)[annotations objectAtIndex:i];
            if([annotation.uniqueId compare:xannotationId] == 0){
                [self removeAnnotation:annotation];
            }
        }
    }
    [xAnnotationsInMapLocker unlock];
    [self refreshAnnotationsInMap];
}



//create annotation
-(MKAnnotationView *)mapView:(MKMapView *)mapView viewForAnnotation:(id<MKAnnotation>)annotation
{
    //handle customized pin
    if(xdelegate != nil){
        MKAnnotationView * annotationView = [xdelegate xmapView:mapView viewForAnnotation:annotation];
        if(annotationView != nil){
            return annotationView;
        }
    }
    
    //user location
    if([annotation isKindOfClass:[MKUserLocation class]]){
        return nil;
    }
    
    //use the XMapView pin
    XMapAnnotationView * pinView = nil;
    if([annotation isKindOfClass:[XMapAnnotation class]] && ![annotation isKindOfClass:[MKUserLocation class]])
    {
        XMapAnnotation * _ant = (XMapAnnotation *)annotation;
        static NSString *pinID = @"XMapAnnotationView";
        pinView = (XMapAnnotationView *)[mapView dequeueReusableAnnotationViewWithIdentifier:pinID];
        if ( pinView == nil ){
            pinView = [[XMapAnnotationView alloc]
                       initWithAnnotation:annotation reuseIdentifier:pinID];
        }
        pinView.alpha = 0;
        pinView.iconFontColor = _ant.iconFontColor;
        pinView.periodFlag = _ant.periodFlag;
        pinView.uniqueId = _ant.uniqueId;
        pinView.iconName = _ant.iconName;
        pinView.pinTag = _ant.pinTag;
        pinView.clustringGroupName = _ant.clustringGroupName;
        pinView.enableMaskColor = _ant.enableMaskColor;
        pinView.iconMaskColor = _ant.iconMaskColor;
        pinView.pinWidth = _ant.pinWidth;
        pinView.enablePinAnimation = _ant.enablePinAnimation;
        pinView.pinHeight = _ant.pinHeight;
        pinView.pinAnchorX = _ant.pinAnchorX;
        pinView.pinAnchorY = _ant.pinAnchorY;
        pinView.enableCustomizedView = _ant.enableCustomizedView;
        pinView.extraParameters = [[NSMutableDictionary alloc] initWithDictionary:_ant.extraParameters];
        pinView.pinCategory = _ant.pinCategory;
        pinView.bundlePath = _ant.bundlePath;
        pinView.documentPath = _ant.documentPath;
        
        
        
        if(pinView.pinCategory != -1){   // not a clustering pin
            if(pinView.customizedView != nil){
                if(pinView.customizedView != _ant.customizedView){
                    if([pinView.subviews count] >= 2){   //didn't remove the pin.customizedView, need to manually remove it
                        [pinView.customizedView removeFromSuperview];
                        pinView.customizedView = nil;
                    }
                }
            }
            pinView.customizedView = _ant.customizedView;
            [pinView addSubview:pinView.customizedView];
        }
        else{
            if(pinView.customizedView != nil){
                [pinView.customizedView removeFromSuperview];
                pinView.customizedView = nil;
            }
        }
        
        pinView.currentZoomLevel = [self getZoomLevel];
        pinView.canShowCallout = NO;
        [pinView setEnabled: YES];
        [pinView setDraggable: YES];
        [pinView reloadData];
    }
    
    
    return pinView;
}


- (int)getZoomLevel {
    return 21-round(log2(self.region.span.longitudeDelta * MERCATOR_RADIUS * M_PI / (180.0 * self.bounds.size.width)));
}


-(void)XMapAnnotationClustering:(NSMutableArray *)entries
{
    if([entries count] <= 0){[xAnnotationsInMapLocker unlock];return;}
    
    //this is used to identify a new turn clustering.
    //set up the flag can help to reduce the unused pin in the map
    NSString * peroidFlag = [NSString stringWithFormat:@"%f",[NSDate timeIntervalSinceReferenceDate]];
    
    NSMutableArray * finalPins = [[NSMutableArray alloc] init];
    NSMutableArray * visitedPins = [[NSMutableArray alloc] init];
    NSMutableArray * groupPins = [[NSMutableArray alloc] init];
    
    CGFloat radius = clusteringRadius;
    if(enableClusteringRadiusAutoAdjust){
        radius = clusteringRadius - [self getZoomLevel] * 2;
    }
    if(radius <= 0){radius = 30;}
    if(radius >= [UIScreen mainScreen].bounds.size.width){radius = [UIScreen mainScreen].bounds.size.width;}
    
    int groupSize = clusteringGroupNrofPins;
    int iTotal = (int)[entries count];
    while ([visitedPins count] <= iTotal)
    {
        if(((int)[entries count]) <= 0){break;}
        XAnnotation * fronter = (XAnnotation *)[entries objectAtIndex:0];
        CLLocationCoordinate2D currentIconCoordinate = CLLocationCoordinate2DMake(fronter.latitude,fronter.longitude);
        CGPoint newCoordinate = [self convertCoordinate:currentIconCoordinate toPointToView:self];
        CGPoint topLeft,bottomRight;
        topLeft.x = (newCoordinate.x - radius)>=0?(newCoordinate.x - radius):0;
        bottomRight.x = (newCoordinate.x + radius)<=self.bounds.size.width?(newCoordinate.x + radius):self.bounds.size.width;
        topLeft.y = (newCoordinate.y - radius)>=0?(newCoordinate.y - radius):0;
        bottomRight.y = (newCoordinate.y + radius)<=self.bounds.size.height?(newCoordinate.y + radius):self.bounds.size.height;
        CLLocationCoordinate2D tmpHolderTL = [self convertPoint:topLeft toCoordinateFromView:self];
        CLLocationCoordinate2D tmpHolderBR = [self convertPoint:bottomRight toCoordinateFromView:self];
        CLLocationCoordinate2D topLeftCoord = CLLocationCoordinate2DMake(MAX(tmpHolderTL.latitude,tmpHolderBR.latitude), MIN(tmpHolderTL.longitude,tmpHolderBR.longitude));
        CLLocationCoordinate2D bottomRightCoord = CLLocationCoordinate2DMake(MIN(tmpHolderTL.latitude,tmpHolderBR.latitude), MAX(tmpHolderTL.longitude,tmpHolderBR.longitude));
        [entries removeObjectsInArray:visitedPins];
        [groupPins removeAllObjects];
        [visitedPins addObject:fronter];
        [groupPins addObject:fronter];
        [entries enumerateObjectsUsingBlock:^(id obj, NSUInteger idx, BOOL *stop) {
            XAnnotation * pinItem = (XAnnotation *)obj;
            CLLocationCoordinate2D cCoordinate = CLLocationCoordinate2DMake(pinItem.latitude, pinItem.longitude);
            if(topLeftCoord.latitude >= cCoordinate.latitude &&  bottomRightCoord.latitude <= cCoordinate.latitude)
            {
                if(topLeftCoord.longitude <= cCoordinate.longitude &&  bottomRightCoord.longitude >= cCoordinate.longitude)
                {
                    if(fronter != obj)
                    {
                        if(([fronter.clustringGroupName compare:pinItem.clustringGroupName] == 0 || ignoreGroupTag) && pinItem.canBeClustered)
                        {
                            [groupPins addObject:obj];
                            [visitedPins addObject:obj];
                        }
                    }
                }
            }
        }];
        [entries removeObjectsInArray:visitedPins];
        CLLocationCoordinate2D cIconCoordinate = CLLocationCoordinate2DMake(fronter.latitude,fronter.longitude);
        if([groupPins count] >= groupSize)//clustering
        {
            int iClusteringNum = (int)[groupPins count];
            [groupPins enumerateObjectsUsingBlock:^(id obj, NSUInteger idx, BOOL *stop) {
                XAnnotation * annoItem = (XAnnotation *)obj;
                NSString * annotationId = annoItem.uniqueId;
                XMapAnnotation * clusteringAnnotation = [self XMapAnnotationByUniqueId:annotationId];
                if(clusteringAnnotation != nil)
                {
                    //do not need to let the last step to clear
                    //it will be cleared after finish the animation
                    clusteringAnnotation.periodFlag = peroidFlag;
                    dispatch_async(dispatch_get_main_queue(), ^{
                        if(annoItem.enablePinAnimation){
                            [UIView animateWithDuration:0.1 delay:0 options:UIViewAnimationOptionCurveEaseIn animations:^{
                                clusteringAnnotation.coordinate = cIconCoordinate;
                            } completion:^(BOOL finished) {
                                [self XMapAnnotationRemover:@[annotationId]];
                            }];
                        }else{[self XMapAnnotationRemover:@[annotationId]];}
                    });
                }
            }];
            
            NSMutableArray * clusteringIcons = [self XMapGetClusteringPinsInRegion:topLeftCoord bottomRightCoord:bottomRightCoord];
            
            //get by location
            BOOL addNewClustering = YES;
            if([clusteringIcons count] > 0)
            {
                //find one and use it delete others
                XMapAnnotation * previousAnnotation = (XMapAnnotation *)[clusteringIcons objectAtIndex:0];
                //clustering pin update information or not
                if([previousAnnotation.iconName compare:[NSString stringWithFormat:@"%d",iClusteringNum]] != 0){
                    [clusteringIcons enumerateObjectsUsingBlock:^(id obj, NSUInteger idx, BOOL *stop) {
                        XMapAnnotation * annoClusterIcon = (XMapAnnotation *)obj;
                        [self XMapAnnotationRemover:@[annoClusterIcon.uniqueId]];
                    }];
                    addNewClustering = YES;
                }else{
                    previousAnnotation.periodFlag = peroidFlag;
                    addNewClustering = NO;
                    [clusteringIcons enumerateObjectsUsingBlock:^(id obj, NSUInteger idx, BOOL *stop) {
                        XMapAnnotation * annoClusterIcon = (XMapAnnotation *)obj;
                        if([annoClusterIcon.uniqueId compare:previousAnnotation.uniqueId] != 0){
                            [self XMapAnnotationRemover:@[annoClusterIcon.uniqueId]];
                        }
                    }];
                }
            }
            if(addNewClustering)
            {
                //create clustering pin
                XMapAnnotation * annotation = [[XMapAnnotation alloc] init];
                annotation.periodFlag = peroidFlag;
                annotation.enablePinAnimation = fronter.enablePinAnimation;
                annotation.pinCategory = -1;//clustering pin
                annotation.clustringGroupName = fronter.clustringGroupName;
                annotation.latitude = cIconCoordinate.latitude;
                annotation.longitude = cIconCoordinate.longitude;
                annotation.coordinate = cIconCoordinate;
                annotation.iconName = (iClusteringNum - 1) > 999? @"999":[NSString stringWithFormat:@"%d", iClusteringNum];
                annotation.iconFontColor = fronter.iconFontColor;
                annotation.iconMaskColor = fronter.iconMaskColor;
                annotation.bundlePath = fronter.bundlePath;
                annotation.documentPath = fronter.documentPath;
                [finalPins addObject:annotation];
            }
        }
        else
        {
            [groupPins enumerateObjectsUsingBlock:^(id obj, NSUInteger idx, BOOL *stop) {
                XAnnotation * itemXAnno = (XAnnotation *)obj;
                //add new annotation
                if(![self XMapAnnotationExisted:itemXAnno.uniqueId])
                {
                    XMapAnnotation * annotation = [[XMapAnnotation alloc] init];
                    annotation.periodFlag = peroidFlag;
                    annotation.enablePinAnimation = itemXAnno.enablePinAnimation;
                    annotation.uniqueId = itemXAnno.uniqueId;
                    annotation.latitude = itemXAnno.latitude;
                    annotation.longitude = itemXAnno.longitude;
                    annotation.coordinate = CLLocationCoordinate2DMake(itemXAnno.latitude,itemXAnno.longitude);
                    annotation.iconFontColor = itemXAnno.iconFontColor;
                    annotation.iconName = itemXAnno.iconName;
                    annotation.pinTag = itemXAnno.pinTag;
                    annotation.pinCategory = 0;
                    annotation.clustringGroupName = itemXAnno.clustringGroupName;
                    annotation.enableMaskColor = itemXAnno.enableMaskColor;
                    annotation.iconMaskColor = itemXAnno.iconMaskColor;
                    annotation.pinWidth = itemXAnno.pinWidth;
                    annotation.pinHeight = itemXAnno.pinHeight;
                    annotation.pinAnchorX = itemXAnno.pinAnchorX;
                    annotation.pinAnchorY = itemXAnno.pinAnchorY;
                    annotation.bundlePath = itemXAnno.bundlePath;
                    annotation.documentPath = itemXAnno.documentPath;
                    annotation.enableCustomizedView = itemXAnno.enableCustomizedView;
                    annotation.customizedView = itemXAnno.customizedView;
                    annotation.extraParameters = [[NSMutableDictionary alloc] initWithDictionary:itemXAnno.extraParameters];
                    [finalPins addObject:annotation];
                }
                else{
                    [self XMapAnnotationSetPeriodFlag:itemXAnno.uniqueId periodFlag:peroidFlag];
                }
            }];
        }
        NSAssert([groupPins count] > 0, @"Number of elements in group should never less than zero");
    }
    
    [finalPins enumerateObjectsUsingBlock:^(id obj, NSUInteger idx, BOOL *stop) {
        XMapAnnotation * annotation = (XMapAnnotation *)obj;
        [xAnnotationsInMap addObject:annotation];
    }];
    [finalPins removeAllObjects];
    
    //clear unused pins
    [self XMapClearUnusedPinsByPeriodFlag:peroidFlag];
    
    //add annotation in the map
    [self performSelectorOnMainThread:@selector(addAnnotations:) withObject:xAnnotationsInMap waitUntilDone:YES];
    [xAnnotationsInMapLocker unlock];
}

-(NSMutableArray *)XMapGetClusteringPinsInRegion:(CLLocationCoordinate2D)topLeftCoord bottomRightCoord:(CLLocationCoordinate2D)bottomRightCoord
{
    NSMutableArray * clusteringIcons = [[NSMutableArray alloc] init];
    NSArray * arrAnnotations = nil;
    arrAnnotations = self.annotations;
    [arrAnnotations enumerateObjectsUsingBlock:^(id obj, NSUInteger idx, BOOL *stop) {
        if(obj != self.userLocation)
        {
            XMapAnnotation * cCoordinate = (XMapAnnotation *)obj;
            if(topLeftCoord.latitude >= cCoordinate.latitude &&  bottomRightCoord.latitude <= cCoordinate.latitude)
            {
                if(topLeftCoord.longitude <= cCoordinate.longitude &&  bottomRightCoord.longitude >= cCoordinate.longitude)
                {
                    if((cCoordinate.pinCategory == -1)){
                        [clusteringIcons addObject:cCoordinate];
                    }
                }
            }
        }
    }];
    return clusteringIcons;
}

-(void)XMapAnnotationSetPeriodFlag:(NSString *)uniqueId periodFlag:(NSString *)peroidFlag
{
    NSArray * arrAnnotations = nil;
    arrAnnotations = self.annotations;
    [arrAnnotations enumerateObjectsUsingBlock:^(id obj, NSUInteger idx, BOOL *stop) {
        if(obj != self.userLocation)
        {
            XMapAnnotation * annotation = (XMapAnnotation *)obj;
            if([annotation.uniqueId compare:uniqueId] == 0){
                annotation.periodFlag = peroidFlag;
            }
        }
    }];
}

-(XMapAnnotation *)XMapAnnotationByCoordiateForClusterIcon:(CLLocationCoordinate2D)coordinate
{
    XMapAnnotation __block * annotation_return = nil;
    NSArray * arrAnnotations = nil;
    arrAnnotations = self.annotations;
    [arrAnnotations enumerateObjectsUsingBlock:^(id obj, NSUInteger idx, BOOL *stop) {
        if(obj != self.userLocation)
        {
            XMapAnnotation * annotation = (XMapAnnotation *)obj;
            if(annotation.coordinate.latitude == coordinate.latitude && annotation.coordinate.longitude == coordinate.longitude && (annotation.pinCategory == -1)){
                annotation_return = annotation;
            }
        }
    }];
    return annotation_return;
}

-(XMapAnnotation *)XMapAnnotationByUniqueId:(NSString *)uniqueId
{
    XMapAnnotation __block * annotation_return = nil;
    NSArray * arrAnnotations = self.annotations;
    [arrAnnotations enumerateObjectsUsingBlock:^(id obj, NSUInteger idx, BOOL *stop) {
        if(obj != self.userLocation){
            XMapAnnotation * annotation = (XMapAnnotation *)obj;
            if([annotation.uniqueId compare:uniqueId] == 0){
                annotation_return = annotation;
            }
        }
    }];
    return annotation_return;
}


-(void)XMapAnnotationRemover:(NSArray *)xannotationIds{
    if([[self annotations] count] <= 0 || [xannotationIds count] <= 0){return;}
    
    NSArray * annotations = self.annotations;
    [xannotationIds enumerateObjectsUsingBlock:^(id obj, NSUInteger idx, BOOL *stop) {
        [annotations enumerateObjectsUsingBlock:^(id objx, NSUInteger idxx, BOOL *stopx) {
            if(objx != self.userLocation)
            {
                XMapAnnotation * annotation = (XMapAnnotation *)objx;
                if([annotation.uniqueId compare:(NSString *)obj] == 0){
                    [self performSelectorOnMainThread:@selector(removeAnnotation:) withObject:annotation waitUntilDone:YES];
                    [self XRemoveSinglePin:annotation];
                    [xAnnotationsInMap removeObject:annotation];
                }
            }
        }];
    }];
}

-(void)XMapClearUnusedPinsByPeriodFlag:(NSString *)periodFlag{
    NSArray * annotations = self.annotations;
    if([annotations count] <= 0){return;}
    [annotations enumerateObjectsUsingBlock:^(id obj, NSUInteger idx, BOOL *stop) {
        if(obj != self.userLocation)
        {
            XMapAnnotation * annotation = (XMapAnnotation *)obj;
            if([annotation.periodFlag compare:periodFlag] != 0){
                [self performSelectorOnMainThread:@selector(removeAnnotation:) withObject:annotation waitUntilDone:YES];
                [self XRemoveSinglePin:annotation];
                [xAnnotationsInMap removeObject:annotation];
            }
        }
    }];
}

-(BOOL)XMapAnnotationExisted:(NSString *)uniqueId
{
    BOOL __block exists = NO;
    NSArray * annotations = self.annotations;
    if([annotations count] <= 0){return NO;}
    [annotations enumerateObjectsUsingBlock:^(id obj, NSUInteger idx, BOOL *stop) {
        if(obj != self.userLocation)
        {
            XMapAnnotation * annotation = (XMapAnnotation *)obj;
            if(annotation.uniqueId == nil){return;}
            if([annotation.uniqueId compare:uniqueId] == 0){
                *stop = NO;
                exists = YES;
            }
        }
    }];
    return exists;
}

-(void)XRemoveSinglePin:(XMapAnnotation *)annot
{
    annot.uniqueId = nil;
    annot.pinCategory  = 0;
    annot.iconName = nil;
    annot.latitude = 0;
    annot.longitude = 0;
    annot.pinTag = nil;
    annot.clustringGroupName = nil;
    annot.enableMaskColor = NO;
    annot.iconMaskColor = nil;
    annot.pinWidth = 20;
    annot.pinHeight = 30;
    annot.pinAnchorX = 0.5;
    annot.pinAnchorY = 1;
    annot.enableCustomizedView = NO;
    annot.customizedView = nil;
    annot.bundlePath = nil;
    annot.documentPath = nil;
    [self XClearDictionaryElement:annot.extraParameters];
}

#pragma clear elements for pins
-(void)XClearArrayElements:(NSMutableArray *)elements
{
    if(elements != nil){
        [elements enumerateObjectsUsingBlock:^(id obj, NSUInteger idx, BOOL *stop) {
            if([obj isKindOfClass:[NSMutableArray class]]){
                [self XClearArrayElements:obj];
            }else if([obj isKindOfClass:[NSMutableDictionary class]]){
                [self XClearDictionaryElement:obj];
            }
            [elements removeObject:obj];
            obj = nil;
        }];
    }
}

-(void)XClearDictionaryElement:(NSMutableDictionary *)element
{
    if(element != nil)
    {
        [element enumerateKeysAndObjectsUsingBlock:^(id key, id obj, BOOL *stop) {
            if([obj isKindOfClass:[NSMutableArray class]]){
                [self XClearArrayElements:obj];
            }else if([obj isKindOfClass:[NSMutableDictionary class]]){
                [self XClearDictionaryElement:obj];
            }
            [element removeObjectForKey:key];
            obj = nil;
        }];
    }
}

#pragma Map Delegate
-(void)mapView:(MKMapView *)mapView didAddAnnotationViews:(NSArray *)views
{
    for (MKAnnotationView *view in views)
    {
        if(view.annotation == mapView.userLocation){
            continue;
        }
        
        if(((XMapAnnotationView *)view).pinCategory == -1)
        {
            [[view superview] bringSubviewToFront:view];
        }
        
        [views enumerateObjectsUsingBlock:^(id obj, NSUInteger idx, BOOL *stop) {
            [UIView animateWithDuration:0.3 delay:0 options:UIViewAnimationOptionCurveEaseIn animations:^{
                ((XMapAnnotationView *)obj).alpha = 1;
            } completion:nil];
        }];
    }
    [xdelegate xmapView:mapView didAddAnnotationViews:views];
}

-(MKOverlayRenderer *)mapView:(MKMapView *)mapView rendererForOverlay:(id<MKOverlay>)overlay
{
    if(xdelegate != nil){
        return [xdelegate xmapView:mapView rendererForOverlay:overlay];
    }
    return nil;
}

-(void)mapView:(MKMapView *)mapView didSelectAnnotationView:(MKAnnotationView *)view
{
    [UIView animateWithDuration:0.2 delay:0 options:UIViewAnimationOptionCurveEaseIn animations:^{
        view.transform = CGAffineTransformMakeScale(1.2, 1.2);
        view.frame = CGRectMake(view.frame.origin.x, view.frame.origin.y+1, view.frame.size.width, view.frame.size.height);
        view.alpha = 0.8;
    } completion:^(BOOL finish){
        [UIView animateWithDuration:0.2 delay:0 options:UIViewAnimationOptionCurveEaseOut animations:^{
            view.transform = CGAffineTransformMakeScale(1, 1);
            view.alpha = 1;
            view.frame = CGRectMake(view.frame.origin.x, view.frame.origin.y-1, view.frame.size.width, view.frame.size.height);
        } completion:nil];
    }];
    [mapView deselectAnnotation:view.annotation animated:YES];
    [xdelegate xmapView:mapView didSelectAnnotationView:view];
}

-(void)mapView:(MKMapView *)mapView regionWillChangeAnimated:(BOOL)animated
{
    [xdelegate xmapView:mapView regionWillChangeAnimated:animated];
}

-(void)mapView:(MKMapView *)mapView regionDidChangeAnimated:(BOOL)animated
{
    dispatch_after(dispatch_time(DISPATCH_TIME_NOW, (int64_t)(0.2 * NSEC_PER_SEC)), dispatch_get_main_queue(), ^{
        [self refreshAnnotationsInMap];
    });
    [xdelegate xmapView:mapView regionDidChangeAnimated:animated];
}

- (void)mapView:(MKMapView *)mapView didUpdateUserLocation:(MKUserLocation *)userLocation
{
    [xdelegate xmapView:mapView didUpdateUserLocation:userLocation];
}


@end
